# Copyright (C) 2003-2005 Peter J. Verveer
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met: 
#
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above
#    copyright notice, this list of conditions and the following
#    disclaimer in the documentation and/or other materials provided
#    with the distribution.
#
# 3. The name of the author may not be used to endorse or promote
#    products derived from this software without specific prior
#    written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
# OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
# GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.      

import types
import math
import numarray
import _ni_support
import _nd_image

def _get_output_fourier(output, input, complex_only = False):
    if complex_only:
        type32 = numarray.Complex32
        type64 = numarray.Complex64
    else:
        type32 = numarray.Float32
        type64 = numarray.Float64
    if output == None:
        if input.type() in [numarray.Complex32, numarray.Complex64]:
            output = numarray.zeros(input.shape, type = input.type())
        elif input.type() == numarray.Float32:
            output = numarray.zeros(input.shape, type = type32)
        else:
            output = numarray.zeros(input.shape, type = type64)
        return_value = output
    else:
        return_value = None
    return output, return_value
    

def fourier_shift(input, shift, n = -1, axis = -1, output = None):
    """Fourier filter for shifting an array.

    The array is multiplied with the fourier transform of a shift
    operation.  An output array can optionally be provided. If the
    parameter n is negative, then the input is assumed to be the
    result of a complex fft. If n is larger or equal to zero, the
    input is assumed to be the result of a real fft, and n gives the
    length of the of the array before transformation long the the real
    transform direction. The axis of the real transform is given by
    the axis parameter.
    """
    input = numarray.asarray(input)
    shift = _ni_support._normalize_sequence(shift, input)
    output, return_value = _get_output_fourier(output, input)
    _nd_image.fourier_shift(input, shift, n, axis, output)
    return return_value

def fourier_gaussian(input, sigma, n = -1, axis = -1, output = None):
    """Multi-dimensional Gaussian fourier filter.

    The array is multiplied with the fourier transform of a Gaussian
    kernel. An output array can optionally be provided. If the
    parameter n is negative, then the input is assumed to be the
    result of a complex fft. If n is larger or equal to zero, the
    input is assumed to be the result of a real fft, and n gives the
    length of the of the array before transformation long the the real
    transform direction. The axis of the real transform is given by
    the axis parameter.
    """
    input = numarray.asarray(input)
    sigma = _ni_support._normalize_sequence(sigma, input)
    output, return_value = _get_output_fourier(output, input)
    _nd_image.fourier_gaussian(input, sigma, n, axis, output)
    return return_value


def fourier_uniform(input, size, n = -1, axis = -1, output = None):
    """Multi-dimensional uniform fourier filter.

    The array is multiplied with the fourier transform of a box of
    given size. An output array can optionally be provided. If the
    parameter n is negative, then the input is assumed to be the
    result of a complex fft. If n is larger or equal to zero, the
    input is assumed to be the result of a real fft, and n gives the
    length of the of the array before transformation long the the real
    transform direction. The axis of the real transform is given by
    the axis parameter.
    """
    input = numarray.asarray(input)
    size = _ni_support._normalize_sequence(size, input)
    output, return_value = _get_output_fourier(output, input)
    _nd_image.fourier_uniform(input, size, n, axis, output)
    return return_value


def fourier_ellipsoid(input, size, n = -1, axis = -1, output = None):
    """Multi-dimensional ellipsoid fourier filter.

    The array is multiplied with the fourier transform of an ellipsoid
    of given size. An output array can optionally be provided. If the
    parameter n is negative, then the input is assumed to be the
    result of a complex fft. If n is larger or equal to zero, the
    input is assumed to be the result of a real fft, and n gives the
    length of the of the array before transformation long the the real
    transform direction. The axis of the real transform is given by
    the axis parameter. This function is implemented for arrays of
    rank 1, 2, or 3.
    """
    input = numarray.asarray(input)
    size = _ni_support._normalize_sequence(size, input)
    output, return_value = _get_output_fourier(output, input, True)
    _nd_image.fourier_ellipsoid(input, size, n, axis, output)
    return return_value
